﻿using System.Runtime.InteropServices;
using System.Collections.Generic;
using System.IO;

namespace SecurePropExampleCs
{
  class Program
  {
    // The name of the dll file to load
    private const string kDllName = "SecureProp.dll";

    // Match the calling convention of the "GetRevealedTextLength" exported function
    [DllImport(kDllName, CharSet = CharSet.Unicode)]
    private static extern uint GetRevealedTextLength(string aSecuredText);

    // Match the calling convention of the "RevealText" exported function
    [DllImport(kDllName, CharSet = CharSet.Unicode)]
    private static extern void RevealText(string aSecuredText, string aPlainTextBuffer, uint aBufferLength);

    private static void LoadPropertiesFromIni(ref List<SecuredProperty> aProperties)
    {
      const string kIniName = "SecureProp.ini";

      if (!File.Exists(kIniName))
      {
        // Couldn't load the ini file, so just output an error message
        System.Console.WriteLine("Couldn't load " + kIniName);
        return;
      }

      string[] lines = File.ReadAllLines(kIniName);
      foreach (string line in lines)
      {
        if (line.Contains("="))
        {
          int index = line.IndexOf('=');
          aProperties.Add(new SecuredProperty(line.Substring(0, index), line.Substring(index + 1)));
        }
      }
    }

    static void Main(string[] args)
    {
      // Load the properties from the ini file
      List<SecuredProperty> properties = new List<SecuredProperty>();
      LoadPropertiesFromIni(ref properties);

      if (properties.Count == 0)
        return;

      //------------------------------------------------------------------------------------------
      // Use the SecureProp library
      //------------------------------------------------------------------------------------------
      foreach (SecuredProperty property in properties)
      {
        string propValue = property.mValue;

        //------------------------------------------------------------------------------------------
        // Exported function call
        // Reveal the secured text and return its length
        //------------------------------------------------------------------------------------------
        uint textLength = GetRevealedTextLength(propValue);

        // Allocate a buffer of wide characters with the length: textLength + 1
        // We need to add 1 in order to provide space for the null terminator
        char[] textBuffer = new char[++textLength];
        string textBufferStr = new string(textBuffer);

        //------------------------------------------------------------------------------------------
        // Exported function call
        // Reveal the secured text and store the result in the provided buffer
        //------------------------------------------------------------------------------------------
        RevealText(propValue, textBufferStr, textLength);

        // Output the revealed text
        System.Console.WriteLine(property.mName + " = " + textBufferStr);
      }

      System.Console.ReadKey();
    }
  }
}
